<?php

namespace App\Http\Controllers\Dashboard\Location;

use App\Http\Constants\SharedConstant;
use App\Http\Controllers\Controller;
use App\Http\Requests\Location\CreateRequest;
use App\Http\Requests\Location\DeleteRequest;
use App\Http\Requests\Location\UpdateRequest;
use App\Models\Location;
use Illuminate\Contracts\View\Factory;
use Illuminate\Http\RedirectResponse;
use Illuminate\Http\Request;

class LocationController extends Controller
{
    public function __construct()
    {
        $this->middleware(['role_or_permission:super-admin|create-locations'])->only('store');
        $this->middleware(['role_or_permission:super-admin|show-locations'])->only('show');
        $this->middleware(['role_or_permission:super-admin|update-locations'])->only('update');
        $this->middleware(['role_or_permission:super-admin|list-locations'])->only('list');
        $this->middleware(['role_or_permission:super-admin|delete-locations'])->only('delete');
    }

    const PER_PAGE = 15;

    /**
     * Display a listing of the resource.
     *
     * @return Application|Factory|\Illuminate\Contracts\View\View
     */
    public function index(Request $request)
    {
        $locations = location::latest();

        if (isset($request['search'])) $locations = $locations->where(function ($query) use ($request) {
            $query->where('name', 'LIKE', "%{$request['search']}%")
                ->orWhere('address', 'LIKE', "%{$request['search']}%");
        });

        $locations = $locations->paginate(self::PER_PAGE);

        return view('locations.index', compact('locations'));
    }

    /**
     * Show the form for creating a new resource.
     *
     */
    public function create()
    {
        return view('locations.create');
    }

    /**
     * Store a newly created resource in storage.
     *
     * @param CreateRequest $request
     * @return RedirectResponse
     */
    public function store(CreateRequest $request)
    {
        location::create($request->validated());

        return redirect()->route('locations.index')->with('success', 'تم إضافة الموقع');
    }

    /**
     * Display the specified resource.
     *
     * @param int $id
     * @return Application|Factory|\Illuminate\Contracts\View\View
     */
    public function show($id)
    {
        $location = location::findOrFail($id);

        return view('locations.show', compact('location',));
    }


    /**
     * Update the specified resource in storage.
     *
     * @param UpdateRequest $request
     * @param location $location
     * @return RedirectResponse
     */
    public function update(UpdateRequest $request, location $location)
    {
        $location->update($request->validated());

        return redirect()->route('locations.index')->with('success', 'تم التعديل');
    }

    /**
     * Remove the specified resource from storage.
     *
     * @param int $id
     * @return RedirectResponse
     */
    public function destroy(int $id)
    {
        location::find($id)->delete();

        return redirect()->route('locations.index')->with('success', 'تم الحذف');
    }
}
