<?php

namespace App\Http\Requests\User;

use App\Http\Constants\UserConstant;
use Illuminate\Contracts\Validation\Validator;
use Illuminate\Foundation\Http\FormRequest;
use Illuminate\Http\RedirectResponse;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Redirect;

class CreateRequest extends FormRequest
{
    /**
     * override failedValidation method.
     * @param Validator $validator
     * @return RedirectResponse|void
     */
    protected function failedValidation(Validator $validator)
    {
        if ($validator->fails()) {
            $errors = $validator->errors();
            return Redirect::back()->withErrors($errors);
        }
    }

    /**
     * Determine if the user is authorized to make this request.
     *
     * @return bool
     */
    public function authorize()
    {
        return true;
    }

    /**
     * Get the validation rules that apply to the request.
     *
     * @return array<string, mixed>
     */
    public function rules()
    {
        return [
            'first_name' => 'required|string|min:3|max:100',
            'last_name' => 'required|string|min:3|max:100',
            'email' => 'required|string|email|max:100|unique:users',
            'phone' => 'required|string|min:3|max:100|unique:users',
            'type' => 'required|integer|in:'
                . UserConstant::TYPES['PILGRIM']
                . ',' . UserConstant::TYPES['SUPERVISOR']
                . ',' . UserConstant::TYPES['ADMIN'],
            'status' => 'nullable|integer|in:'
                . UserConstant::STATUSES['IN_ACTIVE']
                . ',' . UserConstant::STATUSES['ACTIVE'],
            'password' => 'required|string|min:8|max:100|confirmed',
            'role_id' => 'nullable|integer',
            'extra_permissions_ids' => 'nullable|array',
            'extra_permissions_ids.*' => 'integer'
        ];
    }

    /**
     * @param \Illuminate\Validation\Validator $validator
     * @return void
     */
    public function withValidator(Validator $validator)
    {
        $errors = [];

        $validator->after(
            function ($validator) use ($errors) {

                //validate case of Supervisor Can only Create Pilgrims.
                if (Auth::user()->type == UserConstant::TYPES['SUPERVISOR'] && $this->get('type') != UserConstant::TYPES['PILGRIM']) {
                    $errors[] = 'المشرفين لا يمكنهم انشاء حساب لغير الحجاج';
                };

                foreach ($errors as $error) {
                    $validator->errors()->add('', $error);
                }
            }
        );

    }
}
