<?php

namespace App\Http\Controllers\Dashboard\CampHall;

use App\Http\Constants\SharedConstant;
use App\Http\Controllers\Controller;
use App\Http\Requests\CampHall\CreateRequest;
use App\Http\Requests\CampHall\DeleteRequest;
use App\Http\Requests\CampHall\UpdateRequest;
use App\Models\Camp;
use App\Models\CampHall;
use Illuminate\Contracts\Foundation\Application;
use Illuminate\Http\RedirectResponse;
use Illuminate\Http\Request;

class CampHallController extends Controller
{
    public function __construct()
    {
        $this->middleware(['role_or_permission:super-admin|create-camp-halls'])->only('store');
        $this->middleware(['role_or_permission:super-admin|show-camp-halls'])->only('show');
        $this->middleware(['role_or_permission:super-admin|update-camp-halls'])->only('update');
        $this->middleware(['role_or_permission:super-admin|list-camp-halls'])->only('list');
        $this->middleware(['role_or_permission:super-admin|delete-camp-halls'])->only('delete');
    }

    const PER_PAGE = 15;

    /**
     * Display a listing of the resource.
     *
     * @return Application|\Illuminate\Contracts\View\Factory|\Illuminate\Contracts\View\View
     */
    public function index(Request $request)
    {
        $halls = CampHall::latest();

        if (isset($request['status']) && !in_array($request['status'], [null, 'all'])) $halls = $halls->where('status', $request['status']);

        if (isset($request['camp_id']) && !in_array($request['camp_id'], [null, 'all'])) $halls = $halls->where('camp_id', $request['camp_id']);

        if (isset($request['search'])) $halls = $halls->where(function ($query) use ($request) {
            $query->where('name', 'LIKE', "%{$request['search']}%");
        });

        $camps = Camp::all();

        $halls = $halls->paginate(self::PER_PAGE);

        foreach ($halls as $hall) {
            $hall->camp_name = $hall->camp->name;
        }

        return view('camp_halls.index', compact('halls', 'camps'));
    }

    /**
     * Show the form for creating a new resource.
     *
     * @return Application|\Illuminate\Contracts\View\Factory|\Illuminate\Contracts\View\View
     */
    public function create()
    {
        $camps = Camp::whereStatus(SharedConstant::STATUSES['AVAILABLE'])->get();

        return view('camp_halls.create', compact('camps'));
    }

    /**
     * Store a newly created resource in storage.
     *
     * @param CreateRequest $request
     * @return RedirectResponse
     */
    public function store(CreateRequest $request)
    {
        CampHall::create($request->validated());

        return redirect()->route('camp-halls.index')->with('success', 'تم إضافة القاعة');
    }

    /**
     * Display the specified resource.
     *
     * @param int $id
     * @return Application|\Illuminate\Contracts\View\Factory|\Illuminate\Contracts\View\View
     */
    public function show($id)
    {
        $hall = CampHall::findOrFail($id);

        $camps = Camp::whereStatus(SharedConstant::STATUSES['AVAILABLE'])->get();

        $statuses = SharedConstant::STATUSES;

        return view('camp_halls.show', compact('hall', 'camps', 'statuses'));
    }


    /**
     * Update the specified resource in storage.
     *
     * @param UpdateRequest $request
     * @param CampHall $hall
     * @return RedirectResponse
     */
    public function update(UpdateRequest $request, $id)
    {
        $hall = CampHall::findOrFail($id);

        $hall->update($request->validated());

        return redirect()->route('camp-halls.index')->with('success', 'تم التعديل');
    }

    /**
     * Remove the specified resource from storage.
     *
     * @param DeleteRequest $request
     * @param int $id
     * @return RedirectResponse
     */
    public function destroy(DeleteRequest $request, int $id)
    {
        CampHall::find($id)->delete();

        return redirect()->route('camp-halls.index')->with('success', 'تم الحذف');
    }
}
