<?php

namespace App\Http\Controllers\Dashboard\HallBed;

use App\Http\Constants\SharedConstant;
use App\Http\Controllers\Controller;
use App\Http\Requests\HallBed\CreateRequest;
use App\Http\Requests\HallBed\DeleteRequest;
use App\Http\Requests\HallBed\UpdateRequest;
use App\Models\Camp;
use App\Models\CampHall;
use App\Models\HallBed;
use Illuminate\Contracts\Foundation\Application;
use Illuminate\Http\RedirectResponse;
use Illuminate\Http\Request;

class HallBedController extends Controller
{
    public function __construct()
    {
        $this->middleware(['role_or_permission:super-admin|create-hall-beds'])->only('store');
        $this->middleware(['role_or_permission:super-admin|show-hall-beds'])->only('show');
        $this->middleware(['role_or_permission:super-admin|update-hall-beds'])->only('update');
        $this->middleware(['role_or_permission:super-admin|list-hall-beds'])->only('list');
        $this->middleware(['role_or_permission:super-admin|delete-hall-beds'])->only('delete');
    }

    const PER_PAGE = 15;

    /**
     * Display a listing of the resource.
     *
     * @return Application|\Illuminate\Contracts\View\Factory|\Illuminate\Contracts\View\View
     */
    public function index(Request $request)
    {
        $beds = HallBed::latest();

        if (isset($request['status']) && !in_array($request['status'], [null, 'all'])) $beds = $beds->where('status', $request['status']);

        if (isset($request['camp_id']) && !in_array($request['camp_id'], [null, 'all'])) $beds = $beds->whereHas('hall', function ($query) use ($request) {
            $query->where('camp_id', $request['camp_id']);
        });

        if (isset($request['hall_id']) && !in_array($request['hall_id'], [null, 'all'])) $beds = $beds->where('hall_id', $request['hall_id']);

        if (isset($request['search'])) $beds = $beds->where(function ($query) use ($request) {
            $query->where('name', 'LIKE', "%{$request['search']}%");
        });

        $camps = Camp::with('halls')->whereStatus(SharedConstant::STATUSES['AVAILABLE'])->get();

        $halls = CampHall::whereStatus(SharedConstant::STATUSES['AVAILABLE'])->get();

        $beds = $beds->paginate(self::PER_PAGE);

        foreach ($beds as $bed) {
            $bed->hall_name = $bed->hall->name;
            $bed->camp_name = Camp::whereId($bed->hall->camp_id)->value('name');
        }

        return view('hall_beds.index', compact('beds', 'camps', 'halls'));
    }

    /**
     * Show the form for creating a new resource.
     *
     * @return Application|\Illuminate\Contracts\View\Factory|\Illuminate\Contracts\View\View
     */
    public function create()
    {
        $camps = Camp::with('halls')->whereStatus(SharedConstant::STATUSES['AVAILABLE'])->get();

        $halls = CampHall::whereStatus(SharedConstant::STATUSES['AVAILABLE'])->get();

        return view('hall_beds.create', compact('camps', 'halls'));
    }

    /**
     * Store a newly created resource in storage.
     *
     * @param CreateRequest $request
     * @return RedirectResponse
     */
    public function store(CreateRequest $request)
    {
        HallBed::create($request->validated());

        return redirect()->route('hall-beds.index')->with('success', 'تم إضافة السرير');
    }

    /**
     * Display the specified resource.
     *
     * @param int $id
     * @return Application|\Illuminate\Contracts\View\Factory|\Illuminate\Contracts\View\View
     */
    public function show($id)
    {
        $bed = HallBed::findOrFail($id);

        $camps = Camp::with('halls')->whereStatus(SharedConstant::STATUSES['AVAILABLE'])->get();

        $halls = CampHall::whereStatus(SharedConstant::STATUSES['AVAILABLE'])->get();

        $statuses = SharedConstant::STATUSES;

        return view('hall_beds.show', compact('bed', 'camps', 'halls', 'statuses'));
    }


    /**
     * Update the specified resource in storage.
     *
     * @param UpdateRequest $request
     * @param $id
     * @return RedirectResponse
     */
    public function update(UpdateRequest $request, $id)
    {
        $bed = HallBed::findOrFail($id);

        $bed->update($request->validated());

        return redirect()->route('hall-beds.index')->with('success', 'تم التعديل');
    }

    /**
     * Remove the specified resource from storage.
     *
     * @param DeleteRequest $request
     * @param int $id
     * @return RedirectResponse
     */
    public function destroy(DeleteRequest $request, int $id)
    {
        HallBed::find($id)->delete();

        return redirect()->route('hall-beds.index')->with('success', 'تم الحذف');
    }
}
