<?php

namespace App\Http\Controllers\Dashboard\Housing;

use App\Http\Constants\SharedConstant;
use App\Http\Constants\UserConstant;
use App\Http\Controllers\Controller;
use App\Http\Requests\Housing\AssignRequest;
use App\Http\Requests\Housing\CreateRequest;
use App\Models\Camp;
use App\Models\CampHall;
use App\Models\HallBed;
use App\Models\User;
use App\Models\UserHousing;
use Illuminate\Contracts\Foundation\Application;
use Illuminate\Http\RedirectResponse;
use Illuminate\Http\Request;


class HousingController extends Controller
{
    public function __construct()
    {
        $this->middleware(['role_or_permission:super-admin|create-housings'])->only('store');
        $this->middleware(['role_or_permission:super-admin|show-housings'])->only('show');
        $this->middleware(['role_or_permission:super-admin|update-housings'])->only('update');
        $this->middleware(['role_or_permission:super-admin|list-housings'])->only('list');
        $this->middleware(['role_or_permission:super-admin|delete-housings'])->only('delete');
    }

    /**
     * Display a listing of the resource.
     *
     */
    public function index(Request $request)
    {
        //
    }

    /**
     * Show the form for creating a new resource.
     *
     * @return Application|\Illuminate\Contracts\View\Factory|\Illuminate\Contracts\View\View
     */
    public function create()
    {
        //TODO: select only this year pilgrims.
        $pilgrims = User::whereStatus(UserConstant::STATUSES['ACTIVE'])->whereType(UserConstant::TYPES['PILGRIM'])->get();
        $camps = Camp::whereStatus(SharedConstant::STATUSES['AVAILABLE'])->get();
        $halls = CampHall::whereStatus(SharedConstant::STATUSES['AVAILABLE'])->get();
        $beds = HallBed::whereStatus(SharedConstant::STATUSES['AVAILABLE'])->get();
        $allBeds = HallBed::all()->count();
        $assignedBedsCount = HallBed::whereStatus(SharedConstant::STATUSES['NOT_AVAILABLE'])->count();
        $assignedPilgrimsCount = self::AssignedPilgrimsIds()->count();
        $unAssignedPilgrimsCount = self::getUnAssignedPilgrimsIds()->count();

        return view('housings.create', compact('pilgrims', 'camps', 'halls','assignedBedsCount','allBeds', 'beds','assignedPilgrimsCount','unAssignedPilgrimsCount'));
    }

    /**
     * Store a newly created resource in storage.
     *
     * @param CreateRequest $request
     * @return RedirectResponse
     */
    public function store(CreateRequest $request)
    {
        $pilgrims_ids = $this->getUnAssignedPilgrimsIds();

        $beds_ids = $this->getUnAssignedBedsIds();

        foreach ($beds_ids as $key => $bed_id) {

            if ($pilgrims_ids->count() == $key) break;

            //assign each user to a bed.
            UserHousing::create(['user_id' => $pilgrims_ids[$key], 'bed_id' => $bed_id]);

            //update bed status to be un available.
            HallBed::find($bed_id)->update(['status' => SharedConstant::STATUSES['NOT_AVAILABLE']]);
        }

        return redirect()->route('user-housings.create')->with('success', 'تم تسكين الحجاج بشكل عشوائي');
    }

    /**
     * Store a newly created resource in storage.
     *
     * @param AssignRequest $request
     * @return RedirectResponse
     */
    public function assignUser(AssignRequest $request)
    {
        $this->unAssignCurrentBedHousing($request['pilgrim_id']);

        //assign new bed.
        UserHousing::create(['user_id' => $request['pilgrim_id'], 'bed_id' => $request['bed_id']]);

        //update new bed status.
        HallBed::find($request['bed_id'])->update(['status' => SharedConstant::STATUSES['NOT_AVAILABLE']]);

        return redirect()->route('user-housings.create')->with('success', 'تم تسكين الحاج');
    }

    /**
     * Store a newly created resource in storage.
     *
     * @param AssignRequest $request
     * @return RedirectResponse
     */
    public function resetHousings(Request $request)
    {
        //delete all housings.
        UserHousing::truncate();

        //update all Beds status.
        HallBed::query()->update(['status' => SharedConstant::STATUSES['AVAILABLE']]);

        return redirect()->route('user-housings.create')->with('success', 'تم تفريغ جميع الأسِرَّة');
    }


    /**
     * @return mixed
     */
    public static function getUnAssignedPilgrimsIds()
    {
        return User::whereStatus(UserConstant::STATUSES['ACTIVE'])
            ->whereType(UserConstant::TYPES['PILGRIM'])
            ->doesntHave('housing')
            ->pluck('id');
    }

    /**
     * @return mixed
     */
    public static function AssignedPilgrimsIds()
    {
        return User::whereStatus(UserConstant::STATUSES['ACTIVE'])
            ->whereType(UserConstant::TYPES['PILGRIM'])
            ->has('housing')
            ->pluck('id');
    }


    /**
     * @return mixed
     */
    public static function getUnAssignedBedsIds()
    {
        return HallBed::whereStatus(SharedConstant::STATUSES['AVAILABLE'])
            ->doesntHave('housing')
            ->pluck('id');
    }

    /**
     * Remove the specified resource from storage.
     *
     * @param Request $request
     * @return RedirectResponse
     */
    public function unAssignUser(Request $request)
    {
        $this->unAssignCurrentBedHousing($request['un_assigned_pilgrim_id']);

        return redirect()->route('user-housings.create')->with('success', 'تم حذف التسكين للحاج');
    }

    /**
     * @param $pilgrim_id
     * @return void
     */
    public function unAssignCurrentBedHousing($pilgrim_id): void
    {
        //get current bed housing.
        $current_housing = UserHousing::where(['user_id' => $pilgrim_id])->first();

        if ($current_housing) {
            //update current bed status.
            HallBed::find($current_housing->bed_id)->update(['status' => SharedConstant::STATUSES['AVAILABLE']]);

            //delete housing record.
            $current_housing->delete();
        }
    }


}
