<?php

namespace App\Http\Controllers\Dashboard\User;

use App\Exports\UsersExport;
use App\Http\Constants\UserConstant;
use App\Http\Controllers\Controller;
use App\Http\Requests\Housing\AssignRequest;
use App\Http\Requests\User\CreateRequest;
use App\Http\Requests\User\DeleteRequest;
use App\Http\Requests\User\ImportRequest;
use App\Http\Requests\User\UpdateRequest;
use App\Imports\UsersImport;
use App\Models\User;
use Illuminate\Http\JsonResponse;
use Illuminate\Http\RedirectResponse;
use Illuminate\Http\Request;
use Maatwebsite\Excel\Facades\Excel;
use Maatwebsite\Excel\Validators\ValidationException;
use Spatie\MediaLibrary\MediaCollections\Exceptions\FileDoesNotExist;
use Spatie\MediaLibrary\MediaCollections\Exceptions\FileIsTooBig;
use Spatie\Permission\Models\Permission;
use Spatie\Permission\Models\Role;
use Symfony\Component\HttpFoundation\BinaryFileResponse;


class UserController extends Controller
{
    public function __construct()
    {
        $this->middleware(['role_or_permission:super-admin|create-users'])->only('store');
        $this->middleware(['role_or_permission:super-admin|show-users'])->only('show');
        $this->middleware(['role_or_permission:super-admin|update-users'])->only('update');
        $this->middleware(['role_or_permission:super-admin|list-users'])->only('list');
        $this->middleware(['role_or_permission:super-admin|delete-users'])->only('destroy');
        $this->middleware(['role_or_permission:super-admin|export-users'])->only('export');
        $this->middleware(['role_or_permission:super-admin|import-users'])->only('import');
    }

    /**
     * Display a listing of the resource.
     *
     * @return \Illuminate\Contracts\Foundation\Application|\Illuminate\Contracts\View\Factory|\Illuminate\Contracts\View\View
     */
    public function index(Request $request)
    {
        $users = User::where('id', '!=', 1)->latest();

        if (isset($request['status']) && !in_array($request['status'], [null, 'all'])) $users = $users->where('status', $request['status']);

        if (isset($request['type']) && !in_array($request['type'], [null, 'all'])) $users = $users->where('type', $request['type']);

        if (isset($request['search'])) $users = $users->where(function ($query) use ($request) {
            $query->where('first_name', 'LIKE', "%{$request['search']}%")
                ->orWhere('last_name', 'LIKE', "%{$request['search']}%");
        });

        $users = $users->paginate(UserConstant::PER_PAGE);

        return view('users.index', compact('users'));
    }

    /**
     * Show the form for creating a new resource.
     *
     * @return \Illuminate\Contracts\Foundation\Application|\Illuminate\Contracts\View\Factory|\Illuminate\Contracts\View\View
     */
    public function create()
    {
        $roles = Role::with('permissions')->get();

        $permissions = Permission::all()->groupBy('module');

        return view('users.create', compact('roles', 'permissions'));
    }

    /**
     * Store a newly created resource in storage.
     *
     * @param CreateRequest $request
     * @return RedirectResponse
     * @throws FileDoesNotExist
     * @throws FileIsTooBig
     */
    public function store(CreateRequest $request)
    {
        $user = User::create($request->validated());

        //sync photo.
        if ($request['photo']) {
            $this->syncPhoto($user);
        }

        if (!empty($request['role_id'])) {
            $this->assignRole($user, $request['role_id']);
        }

        if (!empty($request['extra_permissions_ids'])) {
            $this->assignExtraPermissions($user, $request['extra_permissions_ids']);
        }

        return redirect()->route('users.index')->with('success', 'تم إضافة المستخدم');
    }

    /**
     * Display the specified resource.
     *
     * @param int $id
     * @return \Illuminate\Contracts\Foundation\Application|\Illuminate\Contracts\View\Factory|\Illuminate\Contracts\View\View
     */
    public function show($id)
    {
        $user = User::findOrFail($id);

        if ($user->housing) {
            $user->bed_id = $user->housing->bed_id;
            $user->hall_id = $user->housing->bed->hall->id;
            $user->camp_id = $user->housing->bed->hall->camp->id;
        }

        if (!empty($user->roles[0])) $user->role_id = $user->roles[0]->id;

        $user->photo = $user->getFirstMediaUrl();

        $roles = Role::with('permissions')->get();

        $permissions = Permission::all()->groupBy('module');

        $statuses = UserConstant::STATUSES;

        $types = UserConstant::TYPES;

        return view('users.show', compact('user', 'roles', 'permissions', 'statuses', 'types'));
    }

    /**
     * Display the specified resource.
     *
     * @param int $id
     * @return JsonResponse
     */
    public function fetchPilgrimData($id)
    {
        $user = User::findOrFail($id);

        $user->camp_id = !empty($user->housing) ? $user->housing->bed->hall->camp->id : '';
        $user->camp_name = !empty($user->housing) ? $user->housing->bed->hall->camp->name : '';
        $user->hall_id = !empty($user->housing) ? $user->housing->bed->hall->id : '';
        $user->hall_name = !empty($user->housing) ? $user->housing->bed->hall->name : '';
        $user->bed_id = !empty($user->housing) ? $user->housing->bed->id : '';
        $user->bed_name = !empty($user->housing) ? $user->housing->bed->name : '';

        return response()->json($user);
    }

    /**
     * Update the specified resource in storage.
     *
     * @param UpdateRequest $request
     * @param User $user
     * @return \Illuminate\Http\Response
     * @throws FileDoesNotExist
     * @throws FileIsTooBig
     */
    public function update(UpdateRequest $request, User $user)
    {
        $user->update($request->validated());

        //sync photo.
        if ($request['photo']) {
            $this->syncPhoto($user);
        }

        //revoke old roles and permissions.
        $this->revokeOldRolesAndPermissions($user);

        //Assign new roles.
        if (!empty($request['role_id'])) {
            $this->assignRole($user, $request['role_id']);
        }

        //Assign new extra permissions.
        if (!empty($request['extra_permissions_ids'])) {
            $this->assignExtraPermissions($user, $request['extra_permissions_ids']);
        }

        return redirect()->route('users.index')->with('success', 'تم التعديل');
    }

    /**
     * @param DeleteRequest $request
     * @param $id
     * @return RedirectResponse
     */
    public function destroy(DeleteRequest $request, $id)
    {
        User::find($id)->delete();

        return redirect()->route('users.index')->with('success', 'تم الحذف');
    }

    /**
     * Display import/export users data.
     *
     * @return \Illuminate\Contracts\Foundation\Application|\Illuminate\Contracts\View\Factory|\Illuminate\Contracts\View\View
     */
    public function usersData()
    {
        return view('users.data');
    }

    /**
     * export users in file
     * @param Request $request
     * @return BinaryFileResponse
     */
    public function export(Request $request): \Symfony\Component\HttpFoundation\BinaryFileResponse
    {
        return Excel::download(new UsersExport($request['year']), 'pilgrims.xlsx');
    }

    /**
     * @param ImportRequest $request
     * @return RedirectResponse
     */
    public function import(ImportRequest $request)
    {
        if(request()->file('file')){

            Excel::import(new UsersImport, request()->file('file'));

            return redirect()->route('users.index')->with('success', 'تم ادخال بيانات الحجاج');

        } else {
            return redirect()->route('users.data')->with('error', 'برجاء اختيار ملف');
        }
    }

    /**
     * Assign Role.
     * Assign Permissions related to the role.
     * @param $user
     * @param $role_id
     * @return void
     */
    public function assignRole($user, $role_id): void
    {
        $user->syncRoles($role_id);

        $role = Role::find($role_id);

        $user->syncPermissions($role->getPermissionNames());
    }

    /**
     * Assign Extra Permissions.
     * @param User $user
     * @param $extra_permissions_ids
     * @return void
     */
    public function assignExtraPermissions(User $user, $extra_permissions_ids): void
    {
        $user->givePermissionTo($extra_permissions_ids);
    }

    /**
     * @param User $user
     * @return void
     * @throws FileDoesNotExist
     * @throws FileIsTooBig
     */
    public function syncPhoto(User $user): void
    {
        $this->deleteOldPhoto($user);

        $this->addNewPhoto($user);
    }

    /**
     * @param User $user
     * @return void
     */
    public function deleteOldPhoto(User $user): void
    {
        $old_photo = $user->getMedia()->first();
        if ($old_photo) $old_photo->delete();
    }

    /**
     * @param User $user
     * @return void
     * @throws FileDoesNotExist
     * @throws FileIsTooBig
     */
    public function addNewPhoto(User $user): void
    {
        $user->addMediaFromRequest('photo')->toMediaCollection();
    }

    /**
     * @param User $user
     * @return void
     */
    public function revokeOldRolesAndPermissions(User $user): void
    {
        $user->syncRoles([]);
        $user->syncPermissions([]);
    }
}
